<?php
require_once 'config.php';

function get_db() {
    static $db = null;
    if ($db === null) {
        $db = new PDO('sqlite:' . DB_FILE);
        $db->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    }
    return $db;
}

// Simple token cache in file
function get_app_oauth_token() {
    $cache_file = DATA_DIR . '/token_cache.json';
    if (file_exists($cache_file)) {
        $o = json_decode(file_get_contents($cache_file), true);
        if ($o && isset($o['expires_at']) && $o['expires_at'] > time() + 60) {
            return $o['access_token'];
        }
    }

    $url = EBAY_ENV . '/identity/v1/oauth2/token';
    $headers = ["Content-Type: application/x-www-form-urlencoded"];
    $data = http_build_query([
        'grant_type' => 'client_credentials',
        'scope' => OAUTH_SCOPE
    ]);

    $ch = curl_init($url);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, $data);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_USERPWD, EBAY_CLIENT_ID . ':' . EBAY_CLIENT_SECRET);
    curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);

    $resp = curl_exec($ch);
    if ($resp === false) {
        throw new Exception('OAuth request failed: ' . curl_error($ch));
    }
    $info = json_decode($resp, true);
    if (!isset($info['access_token'])) {
        throw new Exception('OAuth failed: ' . $resp);
    }
    $expires = time() + intval($info['expires_in'] ?? 7200);
    file_put_contents($cache_file, json_encode(['access_token' => $info['access_token'], 'expires_at' => $expires]));
    return $info['access_token'];
}

// Marketplace Insights search wrapper — expects 'q' parameter
function marketplace_insights_search($query, $limit = 50) {
    $token = get_app_oauth_token();
    // NOTE: Marketplace Insights endpoints are a limited release. This wrapper assumes the "item_sales/search" method exists.
    $url = EBAY_ENV . '/buy/marketplace_insights/v1/item_sales/search';
    $params = http_build_query(['q' => $query, 'limit' => $limit]);
    $full = $url . '?' . $params;

    $ch = curl_init($full);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_HTTPHEADER, ["Authorization: Bearer $token", "Content-Type: application/json"]);
    $resp = curl_exec($ch);
    if ($resp === false) {
        throw new Exception('Marketplace Insights request failed: ' . curl_error($ch));
    }
    $status = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    if ($status >= 400) {
        throw new Exception("Marketplace Insights error ($status): $resp");
    }
    return json_decode($resp, true);
}

// Simple normalization: strip non-alphanum, lowercase
function normalize_part($part) {
    return strtolower(preg_replace('/[^a-z0-9]/i', '', $part));
}

function compute_profit_metrics($median_price, $local_cost, $fee_pct = DEFAULT_FEE_PERCENT, $shipping = DEFAULT_SHIPPING_COST) {
    $fees = $median_price * $fee_pct;
    $net_rev = $median_price - $fees - $shipping;
    $profit = $net_rev - $local_cost;
    $profit_margin = $net_rev > 0 ? ($profit / $net_rev) : null;
    return [
        'median_price' => $median_price,
        'fees' => round($fees,2),
        'shipping' => round($shipping,2),
        'net_revenue' => round($net_rev,2),
        'local_cost' => round($local_cost,2),
        'profit' => round($profit,2),
        'profit_margin' => $profit_margin === null ? null : round($profit_margin,4)
    ];
}

// Simple caching for queries in DB
function get_cached_query($part_number) {
    $db = get_db();
    $stmt = $db->prepare('SELECT result_json, cached_at FROM queries WHERE part_number = :p');
    $stmt->execute([':p' => $part_number]);
    $r = $stmt->fetch(PDO::FETCH_ASSOC);
    if (!$r) return null;
    if (time() - intval($r['cached_at']) > CACHE_TTL_SECONDS) return null;
    return json_decode($r['result_json'], true);
}

function save_cached_query($part_number, $description, $local_cost, $result) {
    $db = get_db();
    $stmt = $db->prepare('INSERT OR REPLACE INTO queries (id, part_number, description, local_cost, result_json, cached_at) VALUES ((SELECT id FROM queries WHERE part_number = :p), :p, :d, :c, :r, :t)');
    $stmt->execute([':p' => $part_number, ':d' => $description, ':c' => $local_cost, ':r' => json_encode($result), ':t' => time()]);
}

?>